CREATE TABLE vit.streets (
	id SERIAL PRIMARY KEY,
	name VARCHAR NOT NULL
);

CREATE TABLE vit.houses (
	id SERIAL PRIMARY KEY,
	street_id INTEGER REFERENCES vit.streets(id),
	number VARCHAR NOT NULL
);

CREATE TABLE vit.apartments (
	id SERIAL PRIMARY KEY,
	house_id INTEGER REFERENCES vit.houses(id),
	number VARCHAR NOT NULL
);

CREATE TABLE vit.manufacturers (
	id SERIAL PRIMARY KEY,
	name VARCHAR NOT NULL
);

CREATE TABLE vit.models (
	id SERIAL PRIMARY KEY,
	manufacturer_id INTEGER REFERENCES vit.manufacturers(id),
	name VARCHAR NOT NULL
);

CREATE TABLE vit.resources (
	id SERIAL PRIMARY KEY,
	name VARCHAR NOT NULL
);

CREATE TABLE vit.meters (
	id SERIAL PRIMARY KEY,
	apartment_id INTEGER REFERENCES vit.apartments(id),
	resource_id INTEGER REFERENCES vit.resources(id),
	serial_number VARCHAR NOT NULL,
	model_id INTEGER REFERENCES vit.models(id),
	verif_interval INTEGER NOT NULL,
	installation_date DATE NOT NULL,
	dismantling_date DATE
);

CREATE TABLE vit.verifications (
	id SERIAL PRIMARY KEY,
	meter_id INTEGER REFERENCES vit.meters(id),
	verif_date DATE NOT NULL,
	valid_to DATE NOT NULL
);

CREATE TABLE vit.meter_values (
	id SERIAL PRIMARY KEY,
	meter_id INTEGER REFERENCES vit.meters(id),
	date TIMESTAMP WITH TIME ZONE NOT NULL,
	t1_value VARCHAR NOT NULL,
	t2_value VARCHAR,
	t3_value VARCHAR
);

CREATE TABLE vit.apartment_meters (
	id SERIAL PRIMARY KEY,
	meter_id INTEGER REFERENCES vit.meters(id),
	apartment_id INTEGER REFERENCES vit.apartments(id)
);

CREATE TABLE vit.house_meters (
	id SERIAL PRIMARY KEY,
	meter_id INTEGER REFERENCES vit.meters(id),
	house_id INTEGER REFERENCES vit.houses(id)
);

ALTER TABLE vit.streets ADD CONSTRAINT streets_name_key UNIQUE (name);
ALTER TABLE vit.houses ADD CONSTRAINT houses_street_number_key UNIQUE (street_id, number);
ALTER TABLE vit.apartments ADD CONSTRAINT apartments_house_number_key UNIQUE (house_id, number);
ALTER TABLE vit.manufacturers ADD CONSTRAINT manufacturers_name_key UNIQUE (name);
ALTER TABLE vit.resources ADD CONSTRAINT resources_name_key UNIQUE (name);
ALTER TABLE vit.models ADD CONSTRAINT models_name_key UNIQUE (name);
ALTER TABLE vit.meters ADD CONSTRAINT meters_serial_number_key UNIQUE (serial_number);
ALTER TABLE vit.house_meters ADD CONSTRAINT house_meters_meter_house_key UNIQUE (meter_id, house_id);
ALTER TABLE vit.apartment_meters ADD CONSTRAINT apartment_meters_meter_apartment_key UNIQUE (meter_id, apartment_id);
ALTER TABLE vit.verifications ADD CONSTRAINT verifications_meter_date_key UNIQUE (meter_id, verif_date);
ALTER TABLE vit.meter_values ADD CONSTRAINT meter_values_meter_date_key UNIQUE (meter_id, date);


CREATE INDEX idx_streets_name ON vit.streets(name);
CREATE INDEX idx_houses_street_number ON vit.houses(street_id, number);
CREATE INDEX idx_apartments_house_number ON vit.apartments(house_id, number);
CREATE INDEX idx_manufacturers_name ON vit.manufacturers(name);
CREATE INDEX idx_models_manufacturer_name ON vit.models(manufacturer_id, name);
CREATE INDEX idx_resources_name ON vit.resources(name);
CREATE INDEX idx_meters_serial_number ON vit.meters(serial_number);
CREATE INDEX idx_meters_model_id ON vit.meters(model_id);
CREATE INDEX idx_meters_resource_id ON vit.meters(resource_id);
CREATE INDEX idx_verifications_meter_date ON vit.verifications(meter_id, verif_date);
CREATE INDEX idx_meter_values_meter_date ON vit.meter_values(meter_id, date);
CREATE INDEX idx_house_meters_meter_house ON vit.house_meters(meter_id, house_id);
CREATE INDEX idx_apartment_meters_meter_apartment ON vit.apartment_meters(meter_id, apartment_id);

-- ------------------------------------

CREATE OR REPLACE PROCEDURE vit.fill_data()
LANGUAGE plpgsql
AS $$
DECLARE
    house RECORD;
    apart JSONB;
    meter JSONB;
    meter_verif JSONB;
    meter_value JSONB;
    v_street_id INTEGER;
    v_house_id INTEGER;
    v_resource_id INTEGER;
    v_model_id INTEGER;
    v_manufacturer_id INTEGER;
    v_meter_id INTEGER;
    v_apartment_id INTEGER;
    street_cache JSONB := '{}';
    house_cache JSONB := '{}';
    manufacturer_cache JSONB := '{}';
    resource_cache JSONB := '{}';
    model_cache JSONB := '{}';
    meter_cache JSONB := '{}';
    apartment_cache JSONB := '{}';
BEGIN
    FOR house IN
        SELECT
            data->>'s' AS street,
            data->>'n' AS number,
            data->'a' AS apartment_data,
            data->'hm' AS meters
        FROM public.input_data
    LOOP
        IF street_cache ? house.street THEN
            v_street_id := (street_cache->>house.street)::INTEGER;
        ELSE
            INSERT INTO vit.streets (name) VALUES (house.street)
            ON CONFLICT (name) DO UPDATE SET name = EXCLUDED.name
            RETURNING id INTO v_street_id;
            street_cache := jsonb_set(street_cache, ARRAY[house.street], to_jsonb(v_street_id));
        END IF;

        DECLARE
            house_key TEXT := v_street_id || '|' || house.number;
        BEGIN
            IF house_cache ? house_key THEN
                v_house_id := (house_cache->>house_key)::INTEGER;
            ELSE
                INSERT INTO vit.houses (street_id, number) VALUES (v_street_id, house.number)
                ON CONFLICT (street_id, number) DO UPDATE SET number = EXCLUDED.number
                RETURNING id INTO v_house_id;
                house_cache := jsonb_set(house_cache, ARRAY[house_key], to_jsonb(v_house_id));
            END IF;
        END;

        IF house.apartment_data IS NOT NULL AND jsonb_typeof(house.apartment_data) = 'array' THEN
            FOR i IN 0..jsonb_array_length(house.apartment_data) - 1
            LOOP
                apart := house.apartment_data->i;
                DECLARE
                    apart_key TEXT := v_house_id || '|' || (apart->>'n');
                BEGIN
                    IF apartment_cache ? apart_key THEN
                        v_apartment_id := (apartment_cache->>apart_key)::INTEGER;
                    ELSE
                        INSERT INTO vit.apartments (house_id, number) VALUES (v_house_id, apart->>'n')
                        ON CONFLICT (house_id, number) DO UPDATE SET number = EXCLUDED.number
                        RETURNING id INTO v_apartment_id;
                        apartment_cache := jsonb_set(apartment_cache, ARRAY[apart_key], to_jsonb(v_apartment_id));
                    END IF;
                END;
            END LOOP;
        END IF;

        IF house.meters IS NOT NULL AND jsonb_typeof(house.meters) = 'array' THEN
            FOR i IN 0..jsonb_array_length(house.meters) - 1
            LOOP
                meter := house.meters->i;

                IF manufacturer_cache ? (meter->>'man') THEN
                    v_manufacturer_id := (manufacturer_cache->>(meter->>'man'))::INTEGER;
                ELSE
                    INSERT INTO vit.manufacturers (name) VALUES (meter->>'man')
                    ON CONFLICT (name) DO UPDATE SET name = EXCLUDED.name
                    RETURNING id INTO v_manufacturer_id;
                    manufacturer_cache := jsonb_set(manufacturer_cache, ARRAY[meter->>'man'], to_jsonb(v_manufacturer_id));
                END IF;

                IF resource_cache ? (meter->>'res') THEN
                    v_resource_id := (resource_cache->>(meter->>'res'))::INTEGER;
                ELSE
                    INSERT INTO vit.resources (name) VALUES (meter->>'res')
                    ON CONFLICT (name) DO UPDATE SET name = EXCLUDED.name
                    RETURNING id INTO v_resource_id;
                    resource_cache := jsonb_set(resource_cache, ARRAY[meter->>'res'], to_jsonb(v_resource_id));
                END IF;

                DECLARE
                    model_key TEXT := v_manufacturer_id || '|' || (meter->>'mod');
                BEGIN
                    IF model_cache ? model_key THEN
                        v_model_id := (model_cache->>model_key)::INTEGER;
                    ELSE
                        INSERT INTO vit.models (manufacturer_id, name) VALUES (v_manufacturer_id, meter->>'mod')
                        ON CONFLICT (name) DO UPDATE SET name = EXCLUDED.name
                        RETURNING id INTO v_model_id;
                        model_cache := jsonb_set(model_cache, ARRAY[model_key], to_jsonb(v_model_id));
                    END IF;
                END;

                IF meter_cache ? (meter->>'sn') THEN
                    v_meter_id := (meter_cache->>(meter->>'sn'))::INTEGER;
                    UPDATE vit.meters SET
                        apartment_id = COALESCE(v_apartment_id, vit.meters.apartment_id),
                        resource_id = v_resource_id,
                        model_id = v_model_id,
                        verif_interval = (meter->>'vi')::INTEGER,
                        installation_date = (meter->>'id')::DATE,
                        dismantling_date = NULLIF(meter->>'dd', '')::DATE
                    WHERE id = v_meter_id;
                ELSE
                    INSERT INTO vit.meters (
                        apartment_id,
                        resource_id,
                        serial_number,
                        model_id,
                        verif_interval,
                        installation_date,
                        dismantling_date
                    ) VALUES (
                        v_apartment_id,
                        v_resource_id,
                        meter->>'sn',
                        v_model_id,
                        (meter->>'vi')::INTEGER,
                        (meter->>'id')::DATE,
                        NULLIF(meter->>'dd', '')::DATE
                    )
                    RETURNING id INTO v_meter_id;
                    meter_cache := jsonb_set(meter_cache, ARRAY[meter->>'sn'], to_jsonb(v_meter_id));
                END IF;

                INSERT INTO vit.house_meters (meter_id, house_id) VALUES (v_meter_id, v_house_id)
                ON CONFLICT DO NOTHING;

                IF v_apartment_id IS NOT NULL THEN
                    INSERT INTO vit.apartment_meters (meter_id, apartment_id) VALUES (v_meter_id, v_apartment_id)
                    ON CONFLICT DO NOTHING;
                END IF;

                IF meter->'vh' IS NOT NULL AND jsonb_typeof(meter->'vh') = 'array' THEN
                    FOR j IN 0..jsonb_array_length(meter->'vh') - 1
                    LOOP
                        meter_verif := meter->'vh'->j;
                        INSERT INTO vit.verifications (meter_id, verif_date, valid_to) VALUES (
                            v_meter_id,
                            (meter_verif->>'d')::DATE,
                            (meter_verif->>'to')::DATE
                        )
                        ON CONFLICT DO NOTHING;
                    END LOOP;
                END IF;

                IF meter->'v' IS NOT NULL AND jsonb_typeof(meter->'v') = 'array' THEN
                    FOR k IN 0..jsonb_array_length(meter->'v') - 1
                    LOOP
                        meter_value := meter->'v'->k;
                        INSERT INTO vit.meter_values (meter_id, date, t1_value, t2_value, t3_value) VALUES (
                            v_meter_id,
                            (meter_value->>'d')::TIMESTAMPTZ,
                            meter_value->>'t1',
                            meter_value->>'t2',
                            meter_value->>'t3'
                        )
                        ON CONFLICT DO NOTHING;
                    END LOOP;
                END IF;
            END LOOP;
        END IF;
    END LOOP;
END;
$$;

-- ----------------------------------------------------------------------------------

CREATE VIEW vit.values_without_verification AS (
	SELECT
		s.name AS street,
		h.number AS house,
		man.name AS manufacturer,
		mod.name AS model,
		m.serial_number,
		mv.date,
		mv.t1_value,
		mv.t2_value,
		mv.t3_value
	FROM vit.house_meters hm
	JOIN vit.houses h
		ON hm.house_id = h.id
	JOIN vit.streets s
		ON h.street_id = s.id
	JOIN vit.meters m
		ON m.id = hm.meter_id
	JOIN vit.meter_values mv
		ON mv.meter_id = m.id
	JOIN vit.models mod
		ON mod.id = m.model_id
	JOIN vit.manufacturers man
		ON man.id = mod.manufacturer_id
);

CREATE VIEW vit.individual_consumption_by_resource AS (
	SELECT
		s.name AS street,
		h.number AS house,
		r.name AS resource,
		mv.date AS date,
		SUM(mv.t1_value::NUMERIC) AS t1,
		SUM(mv.t2_value::NUMERIC) AS t2,
		SUM(mv.t3_value::NUMERIC) AS t3
	FROM vit.house_meters hm
	JOIN vit.houses h
		ON hm.house_id = h.id
	JOIN vit.streets s
		ON h.street_id = s.id
	JOIN vit.meters m
		ON m.id = hm.meter_id
	JOIN vit.meter_values mv
		ON mv.meter_id = m.id
	JOIN vit.resources r
		ON m.resource_id = r.id
	GROUP BY date, street, house, resource
	ORDER BY street, house, resource, date
);


CREATE VIEW vit.meter_models AS (
	SELECT
		man.name AS manufacturer,
		mod.name AS model,
		COUNT(*) AS count
	FROM vit.meters m
	JOIN vit.models mod
		ON mod.id = m.model_id
	JOIN vit.manufacturers man
		ON man.id = mod.manufacturer_id
	GROUP BY manufacturer, model
	ORDER BY manufacturer, model
);


CREATE VIEW vit.values_bihourly AS (
    SELECT
        s.name AS street,
        h.number AS house,
        man.name AS manufacturer,
        mod.name AS model,
        m.serial_number,
        date_trunc('hour', mv.date) + 
            CASE WHEN EXTRACT(minute FROM mv.date) >= 30 THEN interval '30 minutes' 
                 ELSE interval '0 minutes' 
            END AS date,
        mv.t1_value,
        mv.t2_value,
        mv.t3_value
    FROM vit.house_meters hm
    JOIN vit.houses h ON hm.house_id = h.id
    JOIN vit.streets s ON h.street_id = s.id
    JOIN vit.meters m ON m.id = hm.meter_id
    JOIN vit.meter_values mv ON mv.meter_id = m.id
    JOIN vit.models mod ON mod.id = m.model_id
    JOIN vit.manufacturers man ON man.id = mod.manufacturer_id
    ORDER BY street, house, manufacturer, model, serial_number, date
);

CREATE VIEW vit.monthly_consumption AS (
    SELECT
        s.name AS street,
        h.number AS house,
        r.name AS resource,
        date_trunc('month', mv.date) AS month,
        SUM(mv.t1_value::NUMERIC) AS t1_total,
        SUM(mv.t2_value::NUMERIC) AS t2_total,
        SUM(mv.t3_value::NUMERIC) AS t3_total
    FROM vit.house_meters hm
    JOIN vit.houses h ON hm.house_id = h.id
    JOIN vit.streets s ON h.street_id = s.id
    JOIN vit.meters m ON m.id = hm.meter_id
    JOIN vit.meter_values mv ON mv.meter_id = m.id
    JOIN vit.resources r ON m.resource_id = r.id
    GROUP BY street, house, resource, month
    ORDER BY street, house, resource, month
);